package monitor

import (
	"fmt"
	"log"
	"os/exec"
	"strings"
	"time"

	"github.com/zhentianxiang/systemd-exporter/utils"
)

// 缓存上次 ActiveEnterTimestamp
var lastChangeCache = make(map[string]time.Time)

// 缓存“检测到重启”的时间
var restartDetectedAt = make(map[string]time.Time)

// 重启标志维持的时间
const restartNotifyTTL = 5 * time.Minute

func UpdateServiceStatus(services []string) {
	now := time.Now()

	for _, service := range services {
		status, lastChange, err := getServiceStatus(service)
		if err != nil {
			log.Printf("获取服务 %s 状态失败: %v\n", service, err)
			serviceStatus.WithLabelValues(service).Set(0)
			serviceLastChange.WithLabelValues(service).Set(float64(now.Unix()))
			serviceRestarted.WithLabelValues(service).Set(0)
			continue
		}

		// 设置当前服务状态与上次变更时间
		serviceStatus.WithLabelValues(service).Set(boolToFloat(status))
		serviceLastChange.WithLabelValues(service).Set(float64(lastChange.Unix()))

		// 是否发生重启（ActiveEnterTimestamp 是否比之前晚）
		prev, ok := lastChangeCache[service]
		if ok && lastChange.After(prev) {
			// 检测到服务重启，记录时间
			restartDetectedAt[service] = now
			log.Printf("服务 %s 检测到重启 (上次: %v, 当前: %v)", service, prev, lastChange)
		}

		// 判断是否处于“重启通知有效期”内
		if restartTime, exists := restartDetectedAt[service]; exists && now.Sub(restartTime) <= restartNotifyTTL {
			serviceRestarted.WithLabelValues(service).Set(1)
		} else {
			serviceRestarted.WithLabelValues(service).Set(0)
		}

		// 更新缓存
		lastChangeCache[service] = lastChange
	}
}

func boolToFloat(b bool) float64 {
	if b {
		return 1
	}
	return 0
}

func getServiceStatus(service string) (bool, time.Time, error) {
	cmd := exec.Command("systemctl", "is-active", service)
	output, err := cmd.CombinedOutput()
	if err != nil {
		if _, ok := err.(*exec.ExitError); !ok {
			return false, time.Now(), fmt.Errorf("执行 systemctl is-active 失败: %w", err)
		}
	}

	statusStr := strings.TrimSpace(string(output))
	running := statusStr == "active"

	cmd = exec.Command("systemctl", "show", service, "-p", "ActiveEnterTimestamp")
	output, err = cmd.CombinedOutput()
	if err != nil {
		return running, time.Now(), fmt.Errorf("获取 ActiveEnterTimestamp 失败: %w", err)
	}

	lastChange, err := utils.ParseSystemdTime(string(output))
	if err != nil {
		return running, time.Now(), fmt.Errorf("时间解析失败: %w", err)
	}

	return running, lastChange, nil
}
