package utils

import (
	"fmt"
	"strings"
	"time"
)

// 解析 systemctl 输出中的 ActiveEnterTimestamp 字段
// 示例格式: "ActiveEnterTimestamp=2025-06-13 10:30:45 UTC"
func ParseSystemdTime(timeStr string) (time.Time, error) {
	prefix := "ActiveEnterTimestamp="
	if !strings.HasPrefix(timeStr, prefix) {
		return time.Now(), fmt.Errorf("不合法的时间字段: %s", timeStr)
	}

	// 去掉前缀
	timeStr = strings.TrimSpace(strings.TrimPrefix(timeStr, prefix))
	if timeStr == "n/a" || timeStr == "" {
		return time.Now(), nil // 没有记录变更时间
	}

	// 支持多种格式
	formats := []string{
		"2006-01-02 15:04:05 MST",     // 2025-06-13 10:30:45 UTC
		"Mon 2006-01-02 15:04:05 MST", // Thu 2025-06-13 10:30:45 UTC
	}

	// 尝试解析时间
	for _, format := range formats {
		if parsed, err := time.Parse(format, timeStr); err == nil {
			return parsed, nil
		}
	}

	return time.Now(), fmt.Errorf("时间解析失败: %s", timeStr)
}
